<?php

// Class ini bertanggung jawab untuk koneksi database dan eksekusi query.
class Database {
    private $dbh; // Database Handler
    private $stmt; // Statement

    // Konstruktor akan otomatis dipanggil saat instance dibuat
    public function __construct() {
        // Data koneksi diambil dari config.php
        $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME;
        $options = [
            // Memastikan koneksi persisten untuk efisiensi
            PDO::ATTR_PERSISTENT => true,
            // Mode error: Exception (lebih mudah ditangkap)
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ];

        try {
            $this->dbh = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            // Jika koneksi gagal, tampilkan pesan error
            die('Koneksi Database Gagal: ' . $e->getMessage());
        }
    }

    // Menyiapkan query SQL (prepare statement)
    public function query($query) {
        $this->stmt = $this->dbh->prepare($query);
    }

    // Binding parameter ke query yang telah disiapkan
    public function bind($param, $value, $type = null) {
        if (is_null($type)) {
            switch (true) {
                case is_int($value):
                    $type = PDO::PARAM_INT;
                    break;
                case is_bool($value):
                    $type = PDO::PARAM_BOOL;
                    break;
                case is_null($value):
                    $type = PDO::PARAM_NULL;
                    break;
                default:
                    $type = PDO::PARAM_STR;
            }
        }
        $this->stmt->bindValue($param, $value, $type);
    }

    // Eksekusi query yang sudah disiapkan
    public function execute() {
        return $this->stmt->execute();
    }

    // Mengembalikan semua hasil query dalam bentuk array
    public function resultSet() {
        $this->execute();
        return $this->stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Mengembalikan satu baris hasil query (fetch single row)
    public function single() {
        $this->execute();
        return $this->stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Menghitung jumlah baris yang terpengaruh (untuk INSERT/UPDATE/DELETE)
    public function rowCount() {
        return $this->stmt->rowCount();
    }
    
    // Mengembalikan ID dari baris yang baru saja dimasukkan
    public function lastInsertId() {
        return $this->dbh->lastInsertId();
    }
}